<?php
/* --------------------------------------------------------------
   WithdrawalReader.php 2020-02-06
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\Withdrawal\Repository;

use Doctrine\DBAL\Connection;
use Gambio\Admin\Withdrawal\Exceptions\WithdrawalNotFoundException;
use Gambio\Admin\Withdrawal\Interfaces\WithdrawalId;
use Gambio\Admin\Withdrawal\Interfaces\WithdrawalSqlCriteria;
use Gambio\Admin\Withdrawal\Interfaces\WithdrawalSqlPagination;

/**
 * Class WithdrawalReader
 *
 * @package Gambio\Admin\Withdrawal\Repository
 */
class WithdrawalReader
{
    /**
     * @var Connection
     */
    private $db;
    
    
    /**
     * ConfigurationQbReader constructor.
     *
     * @param Connection $db
     */
    public function __construct(Connection $db)
    {
        $this->db = $db;
    }
    
    
    /**
     * Returns the withdrawal data with provided OD.
     *
     * @param WithdrawalId $id
     *
     * @return array
     *
     * @throws WithdrawalNotFoundException
     */
    public function getById(WithdrawalId $id): array
    {
        $withdrawalData = $this->db->createQueryBuilder()
            ->select('*')
            ->from('withdrawals')
            ->where('withdrawal_id = :id')
            ->setParameter('id', $id->value())
            ->execute()
            ->fetch();
        
        if ($withdrawalData === false) {
            throw WithdrawalNotFoundException::forId($id->value());
        }
        
        return $withdrawalData;
    }
    
    
    /**
     * Returns a list of all withdrawals data.
     *
     * @param WithdrawalSqlCriteria   $criteria
     * @param WithdrawalSqlPagination $pagination
     *
     * @return array
     */
    public function getAll(
        WithdrawalSqlCriteria $criteria,
        WithdrawalSqlPagination $pagination
    ): array {
        $query = $this->db->createQueryBuilder()->select('*')->from('withdrawals');
        $criteria->applyToQuery($query);
        $pagination->applyToQuery($query);
        
        return $query->execute()->fetchAll();
    }
    
    
    /**
     * Returns the number of all withdrawals.
     *
     * @param WithdrawalSqlCriteria $criteria
     *
     * @return int
     */
    public function getTotalCount(WithdrawalSqlCriteria $criteria): int
    {
        $query = $this->db->createQueryBuilder()->select('count(withdrawal_id) as total')->from('withdrawals');
        $criteria->applyToQuery($query);
        
        $result = $query->execute()->fetch();
        
        return (int)$result['total'];
    }
}